/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <ctype.h>

#include "udm_config.h"
#include "udm_common.h"
#include "udm_textlist.h"
#include "udm_parsehtml.h"

#include "udm_utils.h"
#include "udm_url.h"
#include "udm_filter.h"
#include "udm_log.h"
#include "udm_xmalloc.h"
#include "udm_server.h"
#include "udm_hrefs.h"
#include "udm_word.h"
#include "udm_crossword.h"
#include "udm_spell.h"
#include "udm_unicode.h"
#include "udm_unidata.h"
#include "udm_uniconv.h"
#include "udm_sgml.h"
#include "udm_guesser.h"
#include "udm_crc32.h"
#include "udm_vars.h"




/****************************************************************/


int UdmPrepareWords(UDM_AGENT * Indexer,UDM_DOCUMENT * Doc){
	size_t		i;
	const char	*doccset;
	UDM_CHARSET	*doccs;
	UDM_CHARSET	*loccs;
	UDM_CHARSET	*sys_int;
	UDM_CONV	dc_uni;
	UDM_CONV	uni_lc;
	UDM_TEXTLIST	*tlist=&Doc->TextList;
	UDM_VAR		*Sec;
	int		res=UDM_OK;
	int		crc32=0;
	int		crossec;
	
	Sec=UdmVarListFind(&Doc->Sections,"crosswords");
	crossec=Sec?Sec->section:0;
	
	doccset=UdmVarListFindStr(&Doc->Sections,"Charset",NULL);
	if(!doccset||!*doccset)doccset=UdmVarListFindStr(&Doc->Sections,"Default-Charset","iso-8859-1");
	doccs=UdmGetCharSet(doccset);
	if(!doccs)doccs=UdmGetCharSet("iso-8859-1");
	loccs = Doc->lcs;
	if(!loccs)loccs=UdmGetCharSet("iso-8859-1");
	sys_int=UdmGetCharSet("sys-int");
	
	UdmConvInit(&dc_uni,doccs,sys_int,0);
	UdmConvInit(&uni_lc,sys_int,loccs,UDM_RECODE_HTML);
	
	
	/* Now convert everything to UNICODE format and calculate CRC32 */
	
	for(i=0;i<tlist->nitems;i++){
		size_t		srclen;
		size_t		dstlen;
		size_t		reslen;
		char		*src,*dst;
		int		uspace=' ';
		int		*lt,*tok,*ustr;
		UDM_TEXTITEM	*Item=&tlist->Item[i];
		
		
		srclen=strlen(Item->str)+1;	/* with '\0' */
		dstlen=(srclen+1)*sizeof(int);	/* with '\0' */
		
		ustr=(int*)malloc(dstlen);
		
		src=Item->str;
		dst=(char*)ustr;
		reslen=UdmConv(&dc_uni,dst,dstlen,src,srclen);
		UdmSGMLUniUnescape(ustr);
		UdmUniRemoveDoubleSpaces(ustr);
		
		/*
		TODO for clones detection:
		Replace any separators into space to ignore 
		various pseudo-graphics, commas, semicolons
		and so on to improve clone detection quality
		*/
		
		if (strncasecmp(Item->section_name, "url", 3) != 0) /* do not calculate crc32  on url* sections */
		  crc32=UdmCRC32Update(crc32,(char*)ustr,reslen);
		
		if((Sec=UdmVarListFind(&Doc->Sections,Item->section_name))){
			
			/* +4 to avoid attempts to fill the only one  */
			/* last byte with multibyte sequence          */
			/* as well as to add a space between portions */
			
			if(Sec->curlen < Sec->maxlen){
				int cnvres;
				
				if(!Sec->val){
					Sec->val=(char*)malloc(Sec->maxlen+1);
				}else{
					/* Add space */
					UdmConv(&uni_lc,Sec->val+Sec->curlen,1,(char*)(&uspace),sizeof(uspace));
					Sec->curlen+=uni_lc.obytes;
					Sec->val[Sec->curlen]='\0';
				}
				
				src=(char*)ustr;
				srclen=UdmUniLen(ustr)*sizeof(int);
				dstlen=Sec->maxlen-Sec->curlen;
				cnvres=UdmConv(&uni_lc,Sec->val+Sec->curlen,dstlen,src,srclen);
				Sec->curlen+=uni_lc.obytes;
				Sec->val[Sec->curlen]='\0';
				
				if (cnvres<0){
					Sec->curlen=Sec->maxlen;
				}
			}
		}
		
		if(Item->section){
			UdmUniStrToLower(ustr);
			
			for(tok=UdmUniGetToken(ustr,&lt); tok ; tok = UdmUniGetToken(NULL, &lt) ){
				size_t	tlen;				/* Word length          */ 
				int	ures;
				int	uword[UDM_MAXWORDSIZE+1];	/* Word in UNICODE      */
				char	lcsword[UDM_MAXWORDSIZE*7+1];	/* Word in LocalCharset */
			
				tlen=lt-tok;
				if(tlen<=UDM_MAXWORDSIZE){
				
					memcpy(uword,tok,tlen*sizeof(int));
					uword[tlen]=0;
				
					UdmConv(&uni_lc,lcsword,sizeof(lcsword),(char*)uword,sizeof(*uword)*(tlen+1));
					ures=strlen(lcsword);
				
					res=UdmWordListAdd(Doc,lcsword,Item->section);
					if(res!=UDM_OK)break;
				
					if(Item->href && crossec){
						UDM_CROSSWORD cw;
						cw.url=Item->href;
						cw.weight=Sec->section;
						cw.pos=Doc->CrossWords.wordpos;
						cw.word=lcsword;
						UdmCrossListAdd(Doc,&cw);
					}
				}
			}
		}
		
		free(ustr);
		if(res!=UDM_OK)break;
	}
	UdmVarListReplaceInt(&Doc->Sections,"crc32",crc32);
	
	return res;
}


/**************************** Built-in Parsers ***************************/

int UdmParseURLText(UDM_AGENT *A,UDM_DOCUMENT *Doc){
	UDM_TEXTITEM	Item;
	UDM_VAR		*Sec;
	
	Item.href=NULL;
	
	if((Sec=UdmVarListFind(&Doc->Sections,"url.proto"))) {
		char sc[]="url.proto";
		Item.str=Doc->CurURL.schema;
		Item.section=Sec->section;
		Item.section_name=sc;
		UdmTextListAdd(&Doc->TextList,&Item);
	}
	if((Sec=UdmVarListFind(&Doc->Sections,"url.host"))) {
		char sc[]="url.host";
		Item.str=Doc->CurURL.hostname;
		Item.section=Sec->section;
		Item.section_name=sc;
		UdmTextListAdd(&Doc->TextList,&Item);
	}
	if((Sec=UdmVarListFind(&Doc->Sections,"url.path"))) {
		char sc[]="url.path";
		Item.str=Doc->CurURL.path;
		Item.section=Sec->section;
		Item.section_name=sc;
		UdmTextListAdd(&Doc->TextList,&Item);
	}
	if((Sec=UdmVarListFind(&Doc->Sections,"url.file"))) {
		char str[UDM_URLSIZE]="";
		char sc[]="url.file";
		UdmUnescapeCGIQuery(str,Doc->CurURL.filename);
		Item.str=Doc->CurURL.filename;
		Item.section=Sec->section;
		Item.section_name=sc;
		UdmTextListAdd(&Doc->TextList,&Item);
	}
	return UDM_OK;
}

int UdmParseHeaders(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc){
	size_t i;
	UDM_TEXTITEM Item;
	
	Item.href=NULL;
	for(i=0;i<Doc->Sections.nvars;i++){
		char	secname[128];
		UDM_VAR	*Sec;
		snprintf(secname,sizeof(secname),"header.%s",Doc->Sections.Var[i].name);
		secname[sizeof(secname)-1]='\0';
		if((Sec=UdmVarListFind(&Doc->Sections,secname))){
			Item.str=Doc->Sections.Var[i].val;
			Item.section=Sec->section;
			Item.section_name=secname;
			UdmTextListAdd(&Doc->TextList,&Item);
		}
	}
	return UDM_OK;
}

int UdmParseText(UDM_AGENT * Indexer,UDM_DOCUMENT * Doc){
	UDM_TEXTITEM	Item;
	UDM_VAR		*BSec=UdmVarListFind(&Doc->Sections,"body");
	char		sc[]="body";
	
	Item.href=NULL;
	
	if(BSec && Doc->Buf.content){
		char *lt;
		Item.section=BSec->section;
		Item.str=strtok_r(Doc->Buf.content, "\r\n", &lt);
		Item.section_name=sc;
		while(Item.str){
			UdmTextListAdd(&Doc->TextList,&Item);
			Item.str=strtok_r(NULL, "\r\n", &lt);
		}
	}
	return(UDM_OK);
}


static void UdmNextCharB(void *d) {
  UDM_HTMLTOK *t = (UDM_HTMLTOK *)d;
  (t->b)++;
}

static void UdmNextCharE(void *d) {
  UDM_HTMLTOK *t = (UDM_HTMLTOK *)d;
  (t->e)++;
}


int UdmHTMLTOKInit(UDM_HTMLTOK *tag) {
  bzero(tag, sizeof(*tag));
  tag->next_b = &UdmNextCharB;
  tag->next_e = &UdmNextCharE;
  return UDM_OK;
}


const char * UdmHTMLToken(const char * s, const char ** lt,UDM_HTMLTOK *t){

	t->ntoks=0;
	t->s = s;
	t->lt = lt;
	
	if(t->s == NULL && (t->s = *lt) == NULL)
		return NULL;

	if(!*t->s) return NULL;
	
	if(!strncmp(t->s,"<!--",4))t->type=UDM_HTML_COM;
	else	
	if(*t->s=='<')t->type=UDM_HTML_TAG;
	else	t->type=UDM_HTML_TXT;

	switch(t->type){
		case UDM_HTML_TAG:

			for(*lt = t->b = t->s + 1; *t->b; ) {
				const char * valbeg=NULL;
				const char * valend=NULL;
				size_t nt=t->ntoks;
				
				
				/* Skip leading spaces */
				while((*t->b)&&strchr(" \t\r\n",*t->b)) (*t->next_b)(t);

				if(*t->b=='>'){
					*lt = t->b + 1;
					return(t->s);
				}

				/* Skip non-spaces, i.e. name */
				for(t->e = t->b; (*t->e) && !strchr(" =>\t\r\n", *t->e); (*t->next_e)(t));
				
				if(t->ntoks<UDM_MAXTAGVAL)
					t->ntoks++;
				
				t->toks[nt].val=0;
				t->toks[nt].vlen=0;
				t->toks[nt].name = t->b;
				t->toks[nt].nlen = t->e - t->b;

				if(!strncasecmp(t->b,"script",6)) t->script = 1;
				if(!strncasecmp(t->b,"/script",7)) t->script = 0;

				if(*t->e=='>'){
					*lt = t->e + 1;
					return(t->s);
				}

				if(!(*t->e)){
					*lt = t->e;
					return(t->s);
				}
				
				/* Skip spaces */
				while((*t->e) && strchr(" \t\r\n",*t->e))(*t->next_e)(t);
				
				if(*t->e != '='){
					t->b = t->e;
				       *lt = t->b;        /* bug when hang on broken inside tag pages fix */
					continue;
				}
				
				/* Skip spaces */
				for(t->b = t->e + 1; (*t->b) && strchr(" \r\n\t", *t->b); (*t->next_b)(t));
				
				if(*t->b == '"'){
					t->b++;
					
					valbeg = t->b;
					for(t->e = t->b; (*t->e) && (*t->e != '"'); (*t->next_e)(t));
					valend = t->e;
					
					t->b = t->e;
					if(*t->b == '"')(*t->next_b)(t);
				}else
				if(*t->b == '\''){
					t->b++;
					
					valbeg = t->b;
					for(t->e = t->b; (*t->e) && (*t->e != '\''); (*t->next_e)(t));
					valend = t->e;
					
					t->b = t->e;
					if(*t->b == '\'')(*t->next_b)(t);
				}else{
					valbeg = t->b;
					for(t->e = t->b; (*t->e) && !strchr(" >\t\r\n", *t->e);(*t->next_e)(t));
					valend = t->e;
					
					t->b = t->e;
				}
				*lt = t->b;
				t->toks[nt].val=valbeg;
				t->toks[nt].vlen=valend-valbeg;
			}
			break;

		case UDM_HTML_COM: /* comment */
			
			if(!UDM_STRNCASECMP(t->s, "<!--UdmComment-->"))
				t->comment=1;
			else
			if(!UDM_STRNCASECMP(t->s, "<!--/UdmComment-->"))
				t->comment=0;

			for(t->e = t->s; (*t->e) && (strncmp(t->e, "-->", 3)); (*t->next_e)(t));
			if(!strncmp(t->e, "-->", 3)) *lt = t->e + 3;
			else	*lt = t->e;
			break;

		case UDM_HTML_TXT: /* text */
		default:
			/* Special case when script  */
			/* body is not commented:    */
			/* <script> x="<"; </script> */
			/* We should find </script>  */
			
			for(t->e = t->s; *t->e; (*t->next_e)(t)){
				if(*t->e == '<'){
					if(t->script){
						if(!UDM_STRNCASECMP(t->e, "</script>")){
							/* This is when script body  */
							/* is not hidden using <!--  */
							break;
						}else
						if(!UDM_STRNCASECMP(t->e, "<!--")){
							/* This is when script body  */
							/* is hidden but there are   */
							/* several spaces between    */
							/* <SCRIPT> and <!--         */
							break;
						}
					}else{
						break;
					}
				}
			}
			
			*lt = t->e;
			break;
	}
	return t->s;
}


int UdmHTMLParseTag(UDM_HTMLTOK * tag,UDM_DOCUMENT * Doc){
	UDM_TEXTITEM Item;
	int opening;
	char name[128];
	register char * n;
	char *metaname=NULL;
	char *metacont=NULL;
	char *href=NULL;
	char *lang = NULL;
	size_t i;

	if(!tag->ntoks)return(0);
	if(!tag->toks[0].name)return(0);
	if(tag->toks[0].nlen>sizeof(name)-1)return(0);

	strncpy(name,tag->toks[0].name,tag->toks[0].nlen);
	name[tag->toks[0].nlen]='\0';
	
	for(i=0;i<tag->ntoks;i++){
		if(ISTAG(i,"name")){
			metaname=strndup(tag->toks[i].val,tag->toks[i].vlen);
		}else
		if(ISTAG(i,"http-equiv")){
			metaname=strndup(tag->toks[i].val,tag->toks[i].vlen);
		}else
		if(ISTAG(i,"content")){
			metacont=strndup(tag->toks[i].val,tag->toks[i].vlen);
		}else
		if(ISTAG(i,"href")){
			/* A, LINK, AREA*/
			char *y = strndup(tag->toks[i].val,tag->toks[i].vlen);
			href = strdup(UdmTrim(y, " \t\r\n"));
			free(y);
		}else
		if(ISTAG(i,"src")){
			/* IMG, FRAME */
			char *y = strndup(tag->toks[i].val,tag->toks[i].vlen);
			href = strdup(UdmTrim(y, " \t\r\n"));
			free(y);
		}else
		if (ISTAG(i, "lang")) {
			char *y = strndup(tag->toks[i].val,tag->toks[i].vlen);
			lang = strdup(UdmTrim(y, " \t\r\n"));
			for(n = lang; *n; *n = tolower(*n),n++);
			free(y);
		}
	}
	
	for(n=name;*n;*n=tolower(*n),n++);
	
	if(name[0]=='/'){
		opening=0;
		memmove(name,name+1,strlen(name+1)+1);
	}else{
		opening=1;
	}

	/* Let's find tag name in order of frequency */

	if(!strcmp(name,"a")){
		UDM_FREE(tag->lasthref);			/*117941*/
	}else
	if(!strcmp(name,"title"))	tag->title=opening;	/*6192*/
	else
	if(!strcmp(name,"html") && opening && (lang != NULL)) {
		UdmVarListReplaceStr(&Doc->Sections,"Content-Language",lang);
	}else
	if(!strcmp(name,"body")) {
		tag->body=opening;	/*5146*/
		if (opening && (lang != NULL)) {
			UdmVarListReplaceStr(&Doc->Sections,"Content-Language",lang);
		}
	}else
	if((!strcmp(name,"meta"))&&(metaname)&&(metacont)){ 
		char	secname[128]="";
		UDM_VAR	*Sec;
		
		strcpy(secname,"meta.");
		strncat(secname+5,metaname,sizeof(secname)-5);
		secname[sizeof(secname)-1]='\0';
		
		if((!tag->comment)&&(Sec=UdmVarListFind(&Doc->Sections,secname))){
			UdmSGMLUnescape(metacont);
			Item.str=metacont;
			Item.section=Sec->section;
			Item.section_name=secname;
			Item.href = NULL;
			UdmTextListAdd(&Doc->TextList,&Item);
		}
		
		if((!strcasecmp(metaname,"Content-Type"))&&(!UdmVarListFindStr(&Doc->Sections,"Charset",NULL))){
			char *p;
			if((p=strstr(metacont,"charset="))){
				const char *cs = UdmCharsetCanonicalName(UdmTrim(p + 8, " \t"));
				UdmVarListAddStr(&Doc->Sections,"Charset",cs?cs:p+8);
			}
		}else
		if(!strcasecmp(metaname, "Content-Language")) {
			char *l;
			l = strdup(metacont);
			for(n = l; *n; *n = tolower(*n),n++);
			UdmVarListReplaceStr(&Doc->Sections,"Content-Language",l);
			free(l);
		}else
		if(!strcasecmp(metaname,"refresh")){
			/* Format: "10; Url=http://something/" */
			/* URL can be written in different     */
			/* forms: URL, url, Url and so on      */
			
			if((href=strchr(metacont,'='))){
				if((href>=metacont+3)&&(!UDM_STRNCASECMP(href-3,"URL="))){
					href=strdup(href+1);
				}else{
					UDM_FREE(href);
				}
			}
 		}else
		if(!strcasecmp(metaname,"robots")&&(Doc->Spider.use_robots)&&(metacont)){
			char * lt;
			char * rtok;
					
			rtok=strtok_r(metacont," ,\r\n\t",&lt);
			while(rtok){
				if(!strcasecmp(rtok,"ALL")){
					/* Set Server parameters */
					tag->follow=Doc->Spider.follow;
					tag->index=Doc->Spider.index;
				}else
				if(!strcasecmp(rtok,"NONE")){
					tag->follow=UDM_FOLLOW_NO;
					tag->index=0;
				}else
				if(!strcasecmp(rtok,"NOINDEX"))
					tag->index=0;
				else
				if(!strcasecmp(rtok,"NOFOLLOW"))
					tag->follow=UDM_FOLLOW_NO;
				else
				if(!strcasecmp(rtok,"INDEX"))
					tag->index=1;
				else
				if(!strcasecmp(rtok,"FOLLOW"))
					tag->follow=Doc->Spider.follow;
				rtok=strtok_r(NULL," \r\n\t",&lt);
			}
		}
	}
	else	if(!strcmp(name,"script"))	tag->script=opening;
	else	if(!strcmp(name,"style"))	tag->style=opening;
	else	if(!strcmp(name,"noindex"))	tag->comment=opening;
	else	
	if((!strcmp(name,"base"))&&(href)){
		
		UdmVarListAddStr(&Doc->Sections,"base.href",href);
		
		/* Do not add BASE HREF itself into database.      */
		/* It will be used only to compose relative links. */
		UDM_FREE(href);
	}

	if((href)&&(tag->follow!=UDM_FOLLOW_NO)){
		UDM_HREF	Href;
		
		UdmSGMLUnescape(href);
		Href.referrer=UdmVarListFindInt(&Doc->Sections,"Referrer-ID",0);
		Href.stored=0;
		Href.tag=NULL;
		Href.category=NULL;
		Href.hops=1+UdmVarListFindInt(&Doc->Sections,"Hops",0);
		Href.url=href;
		Href.method=UDM_METHOD_GET;
		UdmHrefListAdd(&Doc->Hrefs,&Href);
		
		/* For crosswords */
		UDM_FREE(tag->lasthref);
		tag->lasthref=strdup(href);
	}
	UDM_FREE(metaname);
	UDM_FREE(metacont);
	UDM_FREE(href);
	UDM_FREE(lang);
	
	return 0;
}


#define MAXSTACK	1024

typedef struct {
	size_t len;
	char * ofs;
} UDM_TAGSTACK;

int UdmHTMLParse(UDM_AGENT *Indexer,UDM_DOCUMENT *Doc){
	UDM_HTMLTOK	tag;
#ifdef NOTUSED_FOR_XML
	char		stack[MAXSTACK*8]="";
	char   		*sbot=stack;
	size_t		nstack=0;
	UDM_TAGSTACK	Stack[MAXSTACK];
#endif
	UDM_TEXTITEM	Item;
	const char	*htok;
	const char	*last;
	UDM_VAR		*BSec=UdmVarListFind(&Doc->Sections,"body");
	UDM_VAR		*TSec=UdmVarListFind(&Doc->Sections,"title");
	int		body_sec  = BSec ? BSec->section : 0;
	int		title_sec = TSec ? TSec->section : 0;
	char		scb[]="body";
	char		sct[]="title";
	
	bzero(&Item,sizeof(Item));
	UdmHTMLTOKInit(&tag);
	tag.follow=Doc->Spider.follow;
	tag.index=Doc->Spider.index;

	htok=UdmHTMLToken(Doc->Buf.content,&last,&tag);
	
	while(htok){
		char       *tmp=NULL;
		const char *tmpbeg;
		const char *tmpend;

		switch(tag.type){
			
			case UDM_HTML_COM:

				break;

			case UDM_HTML_TXT:

				for( tmpbeg=htok;   tmpbeg<last && strchr(" \r\n\t",tmpbeg[0]) ; tmpbeg++);
				for( tmpend=last-1; htok<tmpend && strchr(" \r\n\t",tmpend[0]) ; tmpend--);
				if(tmpbeg>=tmpend)break;
				
				tmp=strndup(tmpbeg,(size_t)(tmpend-tmpbeg+1));
				
				if(BSec&&!tag.comment&&tag.body&&!tag.script&&!tag.style){
					Item.href=tag.lasthref;
					Item.str=tmp;
					Item.section=body_sec;
					Item.section_name=scb;
					UdmTextListAdd(&Doc->TextList,&Item);
				}
				if(TSec&&!tag.comment&&tag.title){
					Item.href=NULL;
					Item.str=tmp;
					Item.section=title_sec;
					Item.section_name=sct;
					UdmTextListAdd(&Doc->TextList,&Item);
				}
				free(tmp);
				break;
		
			case UDM_HTML_TAG:
				
#ifdef NOTUSED_FOR_XML
				if(0){
					size_t i;
					char str[100];
					strncpy(str,tag.toks[0].name,tag.toks[0].nlen);
					str[tag.toks[0].nlen]='\0';
					fprintf(stderr,"T=%s\n",str);
					
					switch(tag.toks[0].name[0]){
						case '!':
							
							break;
						case '/':
							for(i=nstack;i>0;i--){
								if(Stack[i-1].len+1==tag.toks[0].nlen){
									if(!strncasecmp(Stack[i-1].ofs,tag.toks[0].name+1,Stack[i-1].len)){
										nstack=i-1;
										sbot=Stack[i-1].ofs;
										if(sbot>stack)sbot--;
										*sbot='\0';
										break;
									}
								}
							}
							break;
						default:
							if(*stack){
								strcat(sbot,".");
								sbot++;
							}
							strncpy(sbot,tag.toks[0].name,tag.toks[0].nlen);
							sbot[tag.toks[0].nlen]='\0';
							Stack[nstack].ofs=sbot;
							Stack[nstack].len=tag.toks[0].nlen;
							nstack++;
							sbot=sbot+strlen(sbot);
						
					}
					fprintf(stderr,"S=%s\n\n",stack);
				}
#endif
				UdmHTMLParseTag(&tag,Doc);
				break;
		}
		htok=UdmHTMLToken(NULL,&last,&tag);
		
	}
	UDM_FREE(tag.lasthref);	
	return UDM_OK;
}
